
#include <nxp/iolpc1766.h>
#include "arm_comm.h"
#include <math.h>

void InitClock(void);

typedef unsigned char  BYTE;
typedef unsigned short WORD;
typedef unsigned long  LONG;

#define PI                    (float)(3.141592654)

static LONG triangle[32];

static void T0_Init(void)
{
    /* Turn on Timer 0 */
    PCONP |=  (1<<1);
  
    /* select MAT0.0 at P1.28 */
    PINSEL3 |= (3<<24); 

    /* 100 usec @ 25 Mhz */
    T0MR0 = 2500; 

    /* interrupt on MR0, reset timer on match 0 */
    T0MCR = 0x0003;
	
    /* toggle MAT0.0 pin on match */
    T0EMR = 0x0031;

    /* Reset Timer */
    T0TCR  = 2;
}


void ADC_Init(void)
{
    /* Turn on the ADC */
    PCONP 	|= (1<<12);	
    AD0CR	=  (1<<2) | (0x4<<8) | (1<<21);
    
    /* Select P0.25 as AD0.2 for input */
    PINSEL1	 = (PINSEL1  & ~(0x3<<18)) | (1<<18);

    /* Disable Pullup and Pulldown resistors */
    PINMODE1 = (PINMODE1 & ~(0x3<<18)) | (0x2<<18);
}

void Start_Converstion(void){
	AD0CR	|=	(1<<24);
	return;
}

LONG Get_ADC_value(void){
	/* Wait for conversion to complete */
	while(!(ADSTAT & (1<<2)));
	/* Return the conversion value */
	return ((ADDR2 >> 4) & (0xFFF));
}

/* Main */
int main (void)
{
    int i, t;

    InitClock();
    T0_Init();
    ADC_Init();
    
    /* Select P0.16 as output */
    FIO0DIR |= (1<<16);
    PINSEL1 &= ~(0x3 << 0);
    
    for(i=0; i!=16; i++) triangle[i] = (((i+1)<<6) - 1)<<6;
    for(i=0; i!=16; i++) triangle[16+i] = (((16-i)<<6) - 64)<<6;

    /* Enable AOUT (P0.26) */
    PINSEL1 |= (2<<20);
    
    /* Power up the GPDMA */
    PCONP |= (1<<29);
    
    /* Enable GPDMA  and sync logic */
    DMACCONFIGURATION = 1;

    /* Select MAT0.0 instead of UART0 Tx */
    DMAREQSEL = (1<<0);
    DMACSYNC  = (1<<8);
    
    
    for (;;)                                       // Loop forever
    {
        FIO0PIN ^= (1 << 16);                  // Toggle P0.16
	Start_Converstion();

        /* Load DMA Channel0 */
        DMACC0SRCADDR  = (LONG) &triangle[0];
        DMACC0DESTADDR = (LONG) &DACR;
        DMACC0LLI      = 0;           // linked lists for ch0
        DMACC0CONTROL  = 32           // transfer size (0 - 11) = 32
                    | (0 << 12)        // source burst size (12 - 14) = 1
                    | (0 << 15)        // destination burst size (15 - 17) = 1
                    | (2 << 18)        // source width (18 - 20) = 32 bit
                    | (2 << 21)        // destination width (21 - 23) = 32 bit
                    | (0 << 24)        // source AHB select (24) = AHB 0
                    | (0 << 25)        // destination AHB select (25) = AHB 0
                    | (1 << 26)        // source increment (26) = increment
                    | (0 << 27)        // destination increment (27) = no increment
                    | (0 << 28)        // mode select (28) = access in user mode
                    | (0 << 29)        // (29) = access not bufferable
                    | (0 << 30)        // (30) = access not cacheable
                    | (0 << 31);       // terminal count interrupt disabled

        DMACC0CONFIGURATION   =  1           // channel enabled (0)
                    | (0 << 1)         // source peripheral (1 - 5) = none
                    | (8 << 6)         // destination request peripheral (6 - 10) = MAT0.0
                    | (1 << 11)        // flow control (11 - 13) = mem to per
                    | (0 << 14)        // (14) = mask out error interrupt
                    | (0 << 15)        // (15) = mask out terminal count interrupt
                    | (0 << 16)        // (16) = no locked transfers
                    | (0 << 18);       // (27) = no HALT
        
	T0TCR = 1;                         // enable timer
        while (DMACC0CONFIGURATION & 1) ;    // wait for the DMA to finish
        T0TCR = 0;                         // disable timer

        t =  Get_ADC_value();
	for (i=0; i<(1000 * t); i++);// Delay
    }
}
    
/*************************************************************************
 * Function Name: InitClock
 * Parameters: void
 * Return: void
 *
 * Description: Initialize PLL and clocks' dividers. Hclk - 300MHz,
 *              Fcckl = 100MHz
 *
 *************************************************************************/
void InitClock(void)
{
  // 1. Init OSC
  SCS_bit.OSCRANGE = 0;
  SCS_bit.OSCEN = 1;
  // 2.  Wait for OSC ready
  while(!SCS_bit.OSCSTAT);
  // 3. Disconnect PLL
  PLL0CON_bit.PLLC = 0;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 4. Disable PLL
  PLL0CON_bit.PLLE = 0;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 5. Select source clock for PLL
  CLKSRCSEL_bit.CLKSRC = 1; // Selects the main oscillator as a PLL clock source.
  // 6. Set PLL settings 300 MHz
  PLL0CFG_bit.MSEL = 25-1;
  PLL0CFG_bit.NSEL = 2-1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 7. Enable PLL
  PLL0CON_bit.PLLE = 1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 8. Wait for the PLL to achieve lock
  while(!PLL0STAT_bit.PLOCK);
  // 9. Set clk divider settings
  CCLKCFG   = 3-1;            // 1/3 Fpll
  PCLKSEL0 = PCLKSEL1 = 0;    // other peripherals 100/4 = 25MHz
  // 10. Connect the PLL
  PLL0CON_bit.PLLC = 1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
}
